#!/bin/sh
# SPDX-License-Identifier: GPL-2.0-only
# Copyright (C) 2025 Linaro Ltd.

set -x

# shellcheck disable=SC1091
. ../../lib/sh-test-lib
OUTPUT="$(pwd)/output"
RESULT_FILE="${OUTPUT}/result.txt"
SCRIPT="$(readlink -f "${0}")"
SCRIPTPATH="$(dirname "${SCRIPT}")"

# Test configuration
TEST_SUITE="smoke"  # smoke, full
TPM_DEVICE="/dev/tpm0"
WORK_DIR="${OUTPUT}/tpm-work"

usage() {
    echo "Usage: ${0} [-s smoke|full]
                      [-d TPM device path]
                      [-h help]
" 1>&2
    exit 0
}

while getopts "s:d:h" arg; do
   case "$arg" in
     s) TEST_SUITE="${OPTARG}";;
     d) TPM_DEVICE="${OPTARG}";;
     h) usage;;
     *)
        usage
        error_msg "Invalid option: ${OPTARG}"
        ;;
  esac
done

# Check prerequisites
check_prerequisites() {
    info_msg "Checking TPM prerequisites..."

    # Check for TPM device
    if [ ! -e "${TPM_DEVICE}" ] && [ ! -e "/dev/tpmrm0" ]; then
        error_msg "No TPM device found (${TPM_DEVICE} or /dev/tpmrm0)"
        return 1
    fi

    if [ -e "/dev/tpmrm0" ]; then
        TPM_DEVICE="/dev/tpmrm0"
        info_msg "Using TPM resource manager: ${TPM_DEVICE}"
    else
        info_msg "Using TPM device: ${TPM_DEVICE}"
    fi

    # Check for tpm2-tools
    if ! command -v tpm2_getcap >/dev/null 2>&1; then
        error_msg "tpm2-tools not found in PATH"
        return 1
    fi

    info_msg "tpm2-tools version: $(tpm2_getcap --version 2>&1 | head -1 || echo 'unknown')"

    return 0
}

# Test: Get TPM capabilities
test_getcap() {
    local test_name="tpm2-tools-getcap"
    info_msg "Testing: ${test_name}"

    if tpm2_getcap properties-fixed > "${OUTPUT}/${test_name}.log" 2>&1; then
        report_pass "${test_name}"
    else
        report_fail "${test_name}"
    fi
}

# Test: Get random bytes
test_getrandom() {
    local test_name="tpm2-tools-getrandom"
    info_msg "Testing: ${test_name}"

    if tpm2_getrandom --hex 32 > "${OUTPUT}/${test_name}.log" 2>&1; then
        # Verify we got 32 bytes (64 hex chars)
        local random_data=$(cat "${OUTPUT}/${test_name}.log")
        if [ ${#random_data} -ge 60 ]; then
            report_pass "${test_name}"
        else
            report_fail "${test_name}-insufficient-data"
        fi
    else
        report_fail "${test_name}"
    fi
}

# Test: Read PCRs
test_pcrread() {
    local test_name="tpm2-tools-pcrread"
    info_msg "Testing: ${test_name}"

    if tpm2_pcrread sha256:0,1,2,3 > "${OUTPUT}/${test_name}.log" 2>&1; then
        report_pass "${test_name}"
    else
        report_fail "${test_name}"
    fi
}

# Test: Create primary key
test_createprimary() {
    local test_name="tpm2-tools-createprimary"
    info_msg "Testing: ${test_name}"

    local ctx_file="${WORK_DIR}/primary.ctx"

    if tpm2_createprimary -C o -c "${ctx_file}" > "${OUTPUT}/${test_name}.log" 2>&1; then
        if [ -f "${ctx_file}" ]; then
            report_pass "${test_name}"
            return 0
        else
            report_fail "${test_name}-no-context-file"
            return 1
        fi
    else
        report_fail "${test_name}"
        return 1
    fi
}

# Test: Create and load a key
test_create_key() {
    local test_name="tpm2-tools-create-key"
    info_msg "Testing: ${test_name}"

    local primary_ctx="${WORK_DIR}/primary.ctx"
    local key_pub="${WORK_DIR}/key.pub"
    local key_priv="${WORK_DIR}/key.priv"
    local key_ctx="${WORK_DIR}/key.ctx"

    # First create primary if it doesn't exist
    if [ ! -f "${primary_ctx}" ]; then
        if ! tpm2_createprimary -C o -c "${primary_ctx}" > "${OUTPUT}/${test_name}-primary.log" 2>&1; then
            report_fail "${test_name}-createprimary-failed"
            return 1
        fi
    fi

    # Create key
    if tpm2_create -C "${primary_ctx}" -G rsa -u "${key_pub}" -r "${key_priv}" > "${OUTPUT}/${test_name}.log" 2>&1; then
        if [ -f "${key_pub}" ] && [ -f "${key_priv}" ]; then
            # Try to load the key
            if tpm2_load -C "${primary_ctx}" -u "${key_pub}" -r "${key_priv}" -c "${key_ctx}" > "${OUTPUT}/${test_name}-load.log" 2>&1; then
                report_pass "${test_name}"
                return 0
            else
                report_fail "${test_name}-load-failed"
                return 1
            fi
        else
            report_fail "${test_name}-no-key-files"
            return 1
        fi
    else
        report_fail "${test_name}"
        return 1
    fi
}

# Test: Sign and verify
test_sign_verify() {
    local test_name="tpm2-tools-sign-verify"
    info_msg "Testing: ${test_name}"

    local primary_ctx="${WORK_DIR}/primary.ctx"
    local key_pub="${WORK_DIR}/sign_key.pub"
    local key_priv="${WORK_DIR}/sign_key.priv"
    local key_ctx="${WORK_DIR}/sign_key.ctx"
    local msg_file="${WORK_DIR}/message.txt"
    local sig_file="${WORK_DIR}/message.sig"

    # Create message
    echo "Test message for TPM signing" > "${msg_file}"

    # Create primary if needed
    if [ ! -f "${primary_ctx}" ]; then
        if ! tpm2_createprimary -C o -c "${primary_ctx}" > "${OUTPUT}/${test_name}-primary.log" 2>&1; then
            report_fail "${test_name}-createprimary-failed"
            return 1
        fi
    fi

    # Create signing key
    if ! tpm2_create -C "${primary_ctx}" -G rsa -u "${key_pub}" -r "${key_priv}" -a "sign|fixedtpm|fixedparent|sensitivedataorigin" > "${OUTPUT}/${test_name}-create.log" 2>&1; then
        report_fail "${test_name}-create-failed"
        return 1
    fi

    # Load key
    if ! tpm2_load -C "${primary_ctx}" -u "${key_pub}" -r "${key_priv}" -c "${key_ctx}" > "${OUTPUT}/${test_name}-load.log" 2>&1; then
        report_fail "${test_name}-load-failed"
        return 1
    fi

    # Sign message
    if ! tpm2_sign -c "${key_ctx}" -g sha256 -o "${sig_file}" "${msg_file}" > "${OUTPUT}/${test_name}-sign.log" 2>&1; then
        report_fail "${test_name}-sign-failed"
        return 1
    fi

    # Verify signature
    if tpm2_verifysignature -c "${key_ctx}" -g sha256 -s "${sig_file}" -m "${msg_file}" > "${OUTPUT}/${test_name}-verify.log" 2>&1; then
        report_pass "${test_name}"
        return 0
    else
        report_fail "${test_name}-verify-failed"
        return 1
    fi
}

# Test: Extend PCR
test_pcr_extend() {
    local test_name="tpm2-tools-pcr-extend"
    info_msg "Testing: ${test_name}"

    # Read PCR 16 before
    local pcr_before=$(tpm2_pcrread sha256:16 2>/dev/null | grep -A1 "sha256:" | tail -1 | awk '{print $2}')

    # Extend PCR 16
    if tpm2_pcrextend 16:sha256=0x0000000000000000000000000000000000000000000000000000000000000000 > "${OUTPUT}/${test_name}.log" 2>&1; then
        # Read PCR 16 after
        local pcr_after=$(tpm2_pcrread sha256:16 2>/dev/null | grep -A1 "sha256:" | tail -1 | awk '{print $2}')

        # Verify it changed
        if [ "${pcr_before}" != "${pcr_after}" ]; then
            report_pass "${test_name}"
        else
            report_fail "${test_name}-pcr-unchanged"
        fi
    else
        report_fail "${test_name}"
    fi
}

# Smoke tests - quick validation
run_smoke_tests() {
    info_msg "Running smoke tests..."

    test_getcap
    test_getrandom
    test_pcrread
    test_createprimary
}

# Full tests - comprehensive validation
run_full_tests() {
    info_msg "Running full test suite..."

    # Run smoke tests first
    run_smoke_tests

    # Additional tests
    test_create_key
    test_sign_verify
    test_pcr_extend
}

# Cleanup
cleanup() {
    info_msg "Cleaning up test artifacts..."
    rm -rf "${WORK_DIR}"
}

# Main execution
! check_root && error_msg "This script must be run as root"
create_out_dir "${OUTPUT}"
mkdir -p "${WORK_DIR}"

info_msg "Starting TPM2-Tools test suite: ${TEST_SUITE}"
info_msg "Output directory: ${OUTPUT}"
info_msg "TPM device: ${TPM_DEVICE}"

# Check prerequisites
if ! check_prerequisites; then
    exit 1
fi

# Run tests based on suite
case "${TEST_SUITE}" in
    smoke)
        run_smoke_tests
        ;;
    full)
        run_full_tests
        ;;
    *)
        error_msg "Invalid test suite: ${TEST_SUITE}"
        usage
        ;;
esac

# Cleanup
cleanup

info_msg "TPM2-Tools tests completed"
